<?php
// header.php
// Expect session_start() already called by including page.
?>

<!-- Header markup (drop into your header area) -->
<!DOCTYPE html>
<html>
<head>
  <title>Savithri Shop</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .sticky-header {
      position: sticky;
      top: 0;
      z-index: 1000;
      background: white;
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    /* Add this to your existing CSS */
.header-icon {
    position: relative;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    text-decoration: none;
    color: #333;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.cart-count {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #ff6b6b;
    color: white;
    width: 18px;
    height: 18px;
    border-radius: 50%;
    font-size: 0.6rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 2px solid white;
    display: none;
}

.cart-count.active {
    display: flex;
}
  </style>
</head>
<body>
<div class="sticky-header p-3 d-flex justify-content-between align-items-center">
  <div><img src="images/logo.jpeg" alt="Logo" height="40"></div>
  <div class="d-flex align-items-center">
   <?php if(isset($_SESSION['user_id'])): ?>
     <span class="me-3">Welcome, <?php echo htmlspecialchars($_SESSION['name']); ?></span>

    <?php endif; ?>
    <a href="cart.php" class="header-icon me-3">
      <i class="fas fa-shopping-cart"></i>
      <span class="cart-count" id="cartCount">0</span>
    </a>
    <a href="home.php">
      <img src="images/home-button.png" alt="Home" style="width:30px;height:30px;">
    </a>
  </div>
</div>
<!-- Live cart JS: fetches count, updates badge, listens for events -->
<script>
(function() {
  const cartCountEl = document.getElementById('cartCount');
  const CART_POLL_INTERVAL = 15000; // every 15s

  // Unified: Update badge visually
  function setBadge(count) {
    if (!cartCountEl) return;
    const validCount = Number.isFinite(count) && count > 0 ? count : 0;

    if (validCount <= 0) {
      cartCountEl.style.display = 'none';
      cartCountEl.textContent = '0';
      cartCountEl.classList.remove('active');
    } else {
      cartCountEl.style.display = 'flex';
      cartCountEl.textContent = validCount > 9 ? '9+' : String(validCount);
      cartCountEl.classList.add('active');
    }
  }

  // Fetch count from server API
  async function fetchCount() {
    try {
      const res = await fetch('get_cart_count.php', { credentials: 'same-origin' });
      if (!res.ok) throw new Error('Network');
      const json = await res.json();
      const count = Number(json?.count ?? 0);
      if (!Number.isFinite(count)) throw new Error('Invalid count');
      setBadge(count);
      cacheCount(count);
    } catch (err) {
      const cached = getCachedCount();
      setBadge(cached);
    }
  }

  // Local storage helpers
  const STORAGE_KEY = 'cart_sync_count';

  function cacheCount(count) {
    try {
      localStorage.setItem(STORAGE_KEY, String(count || 0));
    } catch (e) {}
  }

  function getCachedCount() {
    try {
      const val = Number(localStorage.getItem(STORAGE_KEY) || 0);
      return Number.isFinite(val) ? val : 0;
    } catch (e) {
      return 0;
    }
  }

  // Listen for page-triggered updates (add_to_cart etc.)
  document.addEventListener('cartUpdated', (e) => {
    const newCount = Number(e.detail?.count ?? 0);
    setBadge(newCount);
    cacheCount(newCount);
    broadcastCount(newCount);
  });

  // BroadcastChannel for cross-tab sync
  let bc;
  try {
    bc = new BroadcastChannel('cart_channel');
    bc.onmessage = (ev) => {
      if (ev?.data?.count !== undefined) {
        const count = Number(ev.data.count || 0);
        setBadge(count);
        cacheCount(count);
      }
    };
  } catch (e) {
    bc = null;
  }

  // Broadcast fallback via localStorage events
  window.addEventListener('storage', (e) => {
    if (e.key === STORAGE_KEY) {
      const count = Number(e.newValue || 0);
      setBadge(count);
    }
  });

  // Helper to broadcast count updates
  function broadcastCount(count) {
    try {
      if (bc) bc.postMessage({ count });
    } catch (e) {}
  }

  // Public helper function for manual badge update
  window.__updateCartBadge = function(count) {
    const num = Number(count || 0);
    setBadge(num);
    cacheCount(num);
    broadcastCount(num);
  };

  // Initialize on page load
  setBadge(0); // start clean
  fetchCount(); // fetch fresh value
  setInterval(fetchCount, CART_POLL_INTERVAL); // periodic refresh
})();
</script>
