<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include database connection
include 'db.php';
include 'sidebar.php';

// Fetch branches for the dropdown
$branchStmt = $pdo->query("SELECT branch_id, branch_name FROM branch_master");
$branches = $branchStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch products for the dropdown
$productStmt = $pdo->query("SELECT product_id, product_name, mrp FROM product_master");
$products = $productStmt->fetchAll(PDO::FETCH_ASSOC);

// Get current date
$currentDate = date('Y-m-d');

// Fetch the latest sale code and sale_id
$lastSaleStmt = $pdo->prepare("SELECT MAX(CAST(SUBSTRING(sale_code, 3) AS UNSIGNED)) AS last_sale_code FROM branch_sales");
$lastSaleStmt->execute();
$lastSale = $lastSaleStmt->fetch(PDO::FETCH_ASSOC);
$lastNumber = $lastSale['last_sale_code'] ?? 0;

$newSaleCode = 'BR' . str_pad($lastNumber + 1, 5, '0', STR_PAD_LEFT);

$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Fetch POST data
    $branch_id = $_POST['branch_id'] ?? null;
    $sale_date = $_POST['sale_date'] ?? date('Y-m-d');
    $sale_type = $_POST['sale_type'] ?? null;
    $receipt_status = $_POST['receipt_status'] ?? null;
    $quantity = $_POST['quantity'] ?? [];
    $total_price = $_POST['total_price'] ?? 0;

    // Initialize arrays for book details
    $book_ids = $_POST['book_name'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $prices = $_POST['price'] ?? [];
    $amounts = $_POST['amount'] ?? [];

    // Ensure book data is valid
    if (empty($book_ids) || count($quantities) !== count($book_ids)) {
        $message = "Please ensure all book rows are filled out correctly.";
    } else {
        try {
            $pdo->beginTransaction(); // Start a transaction

            // Loop through each book row
            foreach ($book_ids as $index => $book_id) {
                $quantity = $quantities[$index];
                $price = $prices[$index];
                $amount = $amounts[$index];

                $query = "
                    INSERT INTO branch_sales 
                    (sale_code, sale_date, product_id, quantity, total_price, sale_type, receipt_status, branch_name) 
                    VALUES 
                    (:sale_code, :sale_date, :product_id, :quantity, :total_price, :sale_type, :receipt_status, :branch_name)
                ";
                $stmt = $pdo->prepare($query);
                $stmt->execute([
                    ':sale_code' => $newSaleCode,
                    ':sale_date' => $sale_date,
                    ':product_id' => $book_id,
                    ':quantity' => $quantity,
                    ':total_price' => $amount,
                    ':sale_type' => $sale_type,
                    ':receipt_status' => $receipt_status,
                    ':branch_name' => $branches[$branch_id-1]['branch_name'], // Get branch name from dropdown
                ]);
            }

            $pdo->commit(); // Commit the transaction
            $message = 'Sale transaction added successfully!';
        } catch (Exception $e) {
            $pdo->rollBack(); // Rollback on error
            $message = 'Error: ' . $e->getMessage();
        }
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Branch Sale Transaction</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; }
        h1 { margin-top: 20px; }
        .form-label { font-weight: bold; }
        .header-info { background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .table-responsive { margin-top: 20px; }
        .btn-danger, .btn-primary { margin: 0; }
        .btn-danger { font-size: 0.8rem; }
        .radio-toggle { margin-bottom: 20px; }
        .radio-toggle .form-check-inline { margin-right: 15px; }
    </style>
</head>
<body>
<div class="container">
    <h1 class="text-center">Add Branch Sale Transaction</h1>
    
    <!-- Header Section -->
    <div class="header-info d-flex justify-content-between">
        <div><strong>Sale Code:</strong> <?= $newSaleCode ?></div>
    </div>

    <!-- Success/Error Message -->
    <?php if ($message): ?>
        <div class="alert alert-success text-center" role="alert">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <!-- Form Section -->
    <form action="" method="POST">
        <div class="card p-4 mb-3 shadow-sm">
            <h3 class="mb-3 text-secondary">Sales Data</h3>
            <div class="row g-3">
                <div class="col-md-4">
                    <label for="sale_date" class="form-label">Sale Date</label>
                    <input type="date" id="sale_date" name="sale_date" class="form-control" value="<?= $currentDate ?>" required>
                </div>
                <div class="col-md-4">
                    <label for="branch_id" class="form-label">Branch Name</label>
                    <select id="branch_id" name="branch_id" class="form-control" required>
                        <option value="">Select Branch</option>
                        <?php foreach ($branches as $branch): ?>
                            <option value="<?= $branch['branch_id'] ?>"><?= $branch['branch_name'] ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-4">
                    <label for="sale_type" class="form-label">Payment Type</label>
                    <select id="sale_type" name="sale_type" class="form-control" required>
                        <option value="">Select Payment Type</option>
                        <option value="cash">Cash</option>
                        <option value="card">Card</option>
                    </select>
                </div>
                <div class="col-md-4">
                    <label for="receipt_status" class="form-label">Receipt Status</label>
                    <select id="receipt_status" name="receipt_status" class="form-control" required>
                        <option value="paid">Paid</option>
                        <option value="unpaid">Unpaid</option>
                    </select>
                </div>
            </div>
        </div>

        <!-- Book List Section -->
        <div class="card p-4 shadow-sm">
            <h3 class="mb-3 text-secondary">Book List</h3>
            <div class="table-responsive">
                <table class="table table-bordered table-striped" id="bookTable">
                    <thead class="table-primary">
                    <tr>
                        <th>S.No</th>
                        <th>Book Name</th>
                        <th>Quantity</th>
                        <th>Price</th>
                        <th>Amount</th>
                        <th>Action</th>
                    </tr>
                    </thead>
                    <tbody>
                    <tr>
                        <td>1</td>
                        <td>
                            <select name="book_name[]" class="form-control bookName" required>
                                <option value="">Select Book</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?= $product['product_id'] ?>" data-price="<?= $product['mrp'] ?>">
                                        <?= $product['product_name'] ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                        <td>
                            <input type="number" name="quantity[]" class="form-control quantity" min="1" value="1" required>
                        </td>
                        <td>
                            <input type="text" name="price[]" class="form-control price" value="0.00" readonly>
                        </td>
                        <td>
                            <input type="text" name="amount[]" class="form-control amount" value="0.00" readonly>
                        </td>
                        <td>
                            <button type="button" class="btn btn-danger btn-sm deleteRow">Delete</button>
                        </td>
                    </tr>
                    </tbody>
                </table>
            </div>
            <button type="button" id="addRow" class="btn btn-primary mt-3">Add Book</button>
        </div>

        <!-- Submit Section -->
        <div class="text-end mt-4">
            <button type="submit" class="btn btn-success">Save Sale</button>
        </div>
    </form>
</div>

<script>
    const products = <?= json_encode($products); ?>;

    // Function to update row details
    function updateRow(row) {
        const productSelect = row.querySelector('.bookName');
        const quantityField = row.querySelector('.quantity');
        const priceField = row.querySelector('.price');
        const amountField = row.querySelector('.amount');

        const selectedOption = productSelect.options[productSelect.selectedIndex];
        const price = parseFloat(selectedOption.getAttribute('data-price')) || 0;
        const quantity = parseFloat(quantityField.value) || 0;
        const amount = price * quantity;

        priceField.value = price.toFixed(2);
        amountField.value = amount.toFixed(2);
        updateTotal();
    }

    // Function to update total amount
    function updateTotal() {
        let total = 0;
        document.querySelectorAll('.amount').forEach(amountField => {
            const amount = parseFloat(amountField.value) || 0;
            total += amount;
        });
        document.getElementById('totalAmount').textContent = total.toFixed(2);
    }

    // Event listener for adding rows
    document.getElementById('addRow').addEventListener('click', () => {
        const bookTable = document.getElementById('bookTable').getElementsByTagName('tbody')[0];
        const rowIndex = bookTable.rows.length + 1;

        const newRow = bookTable.insertRow();
        newRow.innerHTML = `
            <td>${rowIndex}</td>
            <td>
                <select name="book_name[]" class="bookName" required>
                    <option value="">Select Book</option>
                    <?php foreach ($products as $product): ?>
                        <option value="<?= $product['product_id'] ?>" data-price="<?= $product['mrp'] ?>">
                            <?= $product['product_name'] ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </td>
            <td>
                <input type="number" name="quantity[]" class="quantity" value="1" min="1" required>
            </td>
            <td>
                <input type="text" name="price[]" class="price" value="0.00" readonly>
            </td>
            <td>
                <input type="text" name="amount[]" class="amount" value="0.00" readonly>
            </td>
            <td>
                <button type="button" class="btn btn-danger btn-sm deleteRow">Delete</button>
            </td>
        `;
    });

    // Event listener for deleting rows
    document.getElementById('bookTable').addEventListener('click', (e) => {
        if (e.target.closest('.deleteRow')) {
            const row = e.target.closest('tr');
            row.remove();
            updateTotal();
        }
    });

    // Event listener for change in quantity or product
    document.getElementById('bookTable').addEventListener('change', (e) => {
        if (e.target.closest('.bookName') || e.target.closest('.quantity')) {
            updateRow(e.target.closest('tr'));
        }
    });
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
